<?php
/**
 * Services Shortcodes Class
 * کلاس شورت‌کدهای خدمات
 */

if (!defined('ABSPATH')) {
    exit;
}

class Services_Shortcodes {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_shortcode('customer_services', array($this, 'display_customer_services'));
        add_shortcode('customer_services_count', array($this, 'display_customer_services_count'));
        add_action('wp_head', array($this, 'add_styles'));
        add_action('wp_footer', array($this, 'add_scripts'));
    }
    
    public function display_customer_services($atts) {
        $atts = shortcode_atts(array(
            'customer_id' => get_the_ID(),
            'posts_per_page' => -1,
            'show_image' => 'yes',
            'show_category' => 'yes',
            'columns' => '4'
        ), $atts);
        
        $services_query = new WP_Query(array(
            'post_type' => 'services',
            'posts_per_page' => intval($atts['posts_per_page']),
            'meta_query' => array(
                array(
                    'key' => 'customer_relation',
                    'value' => $atts['customer_id'],
                    'compare' => '='
                )
            ),
            'post_status' => 'publish'
        ));
        
        if (!$services_query->have_posts()) {
            return '<div class="no-services-simple"><p>' . __('هیچ خدمتی برای این مشتری ثبت نشده است.', 'services-post-type') . '</p></div>';
        }
        
        ob_start();
        ?>
        <div class="customer-services-clean">
            <div class="services-grid-clean services-columns-<?php echo esc_attr($atts['columns']); ?>">
                <?php while ($services_query->have_posts()) : $services_query->the_post(); ?>
                    <?php
                    // بررسی دسته‌بندی خدمت
                    $categories = get_the_terms(get_the_ID(), 'service-category');
                    $is_web_design = false;
                    if ($categories && !is_wp_error($categories)) {
                        foreach ($categories as $category) {
                            if ($category->name === 'طراحی وب') {
                                $is_web_design = true;
                                break;
                            }
                        }
                    }
                    ?>
                    <div class="service-item-clean <?php echo $is_web_design ? 'web-design-item' : ''; ?>">
                        <?php if ($atts['show_image'] === 'yes' && has_post_thumbnail()) : ?>
                            <div class="service-image-clean <?php echo $is_web_design ? 'web-design-image' : ''; ?>">
                                <?php 
                                // نمایش تصویر با کیفیت بالا
                                $thumbnail_id = get_post_thumbnail_id();
                                if ($is_web_design) {
                                    // برای طراحی وب از سایز بزرگ استفاده کن
                                    $image_url = wp_get_attachment_image_url($thumbnail_id, 'service-high-quality');
                                    if (!$image_url) {
                                        $image_url = wp_get_attachment_image_url($thumbnail_id, 'full');
                                    }
                                } else {
                                    // برای سایر خدمات از سایز استاندارد استفاده کن
                                    $image_url = wp_get_attachment_image_url($thumbnail_id, 'service-standard');
                                    if (!$image_url) {
                                        $image_url = wp_get_attachment_image_url($thumbnail_id, 'large');
                                    }
                                }
                                
                                $image_alt = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);
                                
                                if ($image_url) {
                                    echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($image_alt ? $image_alt : get_the_title()) . '" loading="lazy" />';
                                }
                                ?>
                            </div>
                        <?php endif; ?>
                        
                        <div class="service-content-clean">
                            <h4 class="service-title-clean">
                                <?php the_title(); ?>
                            </h4>
                            
                            <?php if ($atts['show_category'] === 'yes') : ?>
                                <?php if ($categories && !is_wp_error($categories)) : ?>
                                    <div class="service-categories-clean">
                                        <?php foreach ($categories as $category) : ?>
                                            <span class="service-category-clean"><?php echo esc_html($category->name); ?></span>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endwhile; ?>
            </div>
        </div>
        
        <?php
        wp_reset_postdata();
        return ob_get_clean();
    }
    
    public function display_customer_services_count($atts) {
        $atts = shortcode_atts(array(
            'customer_id' => get_the_ID(),
            'text' => __('تعداد خدمات: ', 'services-post-type')
        ), $atts);
        
        $count = get_customer_services_count($atts['customer_id']);
        
        return '<span class="services-count">' . esc_html($atts['text']) . '<strong>' . $count . '</strong></span>';
    }
    
    public function add_styles() {
        ?>
        <style>
        .customer-services-clean {
            margin: 30px 0;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
            direction: rtl;
            text-align: right;
        }

        .services-grid-clean {
            display: grid;
            gap: 25px;
            margin-top: 20px;
        }

        .services-columns-1 { grid-template-columns: 1fr; }
        .services-columns-2 { grid-template-columns: repeat(2, 1fr); }
        .services-columns-3 { grid-template-columns: repeat(3, 1fr); }
        .services-columns-4 { grid-template-columns: repeat(4, 1fr); }

        .service-item-clean {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            border: 1px solid #f0f0f0;
            text-align: center;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
            cursor: default;
            position: relative;
        }

        .service-image-clean {
            position: relative;
            overflow: hidden;
            margin-bottom: 0;
            height: 220px;
            background: #f8f9fa;
        }

        .service-image-clean img {
            width: 100%;
            height: 220px;
            object-fit: cover;
            object-position: center;
            display: block;
            image-rendering: -webkit-optimize-contrast;
            image-rendering: optimize-contrast;
            image-rendering: crisp-edges;
            -ms-interpolation-mode: bicubic;
            transition: transform 0.3s ease;
        }

        .web-design-item {
            position: relative;
        }

        .web-design-image {
            height: 220px;
            overflow: hidden;
            position: relative;
            cursor: pointer;
            background: #fff;
        }

        .web-design-image img {
            width: 100%;
            height: auto;
            min-height: 220px;
            object-fit: cover;
            object-position: top center;
            transition: transform 8s ease-in-out;
            transform: translateY(0);
            image-rendering: -webkit-optimize-contrast;
            image-rendering: optimize-contrast;
            image-rendering: crisp-edges;
            -ms-interpolation-mode: bicubic;
        }

        .web-design-item:hover .web-design-image img {
            transform: translateY(calc(-100% + 220px));
        }

        .service-content-clean {
            padding: 20px 15px;
        }

        .service-title-clean {
            margin: 0 0 12px 0;
            font-size: 16px;
            line-height: 1.5;
            font-weight: bold;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
            color: #333;
        }

        .service-categories-clean {
            margin-top: 10px;
            display: flex;
            justify-content: center;
            flex-wrap: wrap;
            gap: 6px;
        }

        .service-category-clean {
            display: inline-block;
            background: #f5f5f5;
            color: #666;
            padding: 5px 12px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: normal;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
        }

        .web-design-item .service-category-clean {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            font-weight: bold;
        }

        .no-services-simple {
            text-align: center;
            padding: 30px 20px;
            margin: 20px 0;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
        }

        .no-services-simple p {
            color: #666;
            font-size: 16px;
            margin: 0;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
        }

        .services-count {
            background: #e7f3ff;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 14px;
            font-family: 'Dana', 'Tahoma', Arial, sans-serif;
        }

        /* Responsive Design */
        @media (max-width: 1200px) {
            .services-columns-4 { 
                grid-template-columns: repeat(3, 1fr); 
            }
        }

        @media (max-width: 900px) {
            .services-columns-4,
            .services-columns-3 { 
                grid-template-columns: repeat(2, 1fr); 
            }
            
            .services-grid-clean {
                gap: 20px;
            }
            
            .service-image-clean,
            .web-design-image {
                height: 200px;
            }
            
            .service-image-clean img,
            .web-design-image img {
                height: 200px;
                min-height: 200px;
            }
            
            .web-design-item:hover .web-design-image img {
                transform: translateY(calc(-100% + 200px));
            }
        }

        @media (max-width: 600px) {
            .services-columns-4,
            .services-columns-3,
            .services-columns-2 { 
                grid-template-columns: 1fr; 
            }
            
            .services-grid-clean {
                gap: 25px;
            }
            
            .service-image-clean,
            .web-design-image {
                height: 250px;
            }
            
            .service-image-clean img,
            .web-design-image img {
                height: 250px;
                min-height: 250px;
            }
            
            .web-design-item:hover .web-design-image img {
                transform: translateY(calc(-100% + 250px));
            }
            
            .service-title-clean {
                font-size: 15px;
            }
        }
        </style>
        <?php
    }
    
    public function add_scripts() {
        ?>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            // فقط برای موبایل
            if (window.innerWidth <= 768) {
                const webImages = document.querySelectorAll('.web-design-image img');
                
                webImages.forEach(function(img) {
                    // اضافه کردن انیمیشن خودکار
                    img.style.animation = 'mobileScroll 10s ease-in-out infinite';
                });
                
                // تعریف انیمیشن CSS
                const style = document.createElement('style');
                style.textContent = `
                    @keyframes mobileScroll {
                        0% { transform: translateY(0) !important; }
                        50% { transform: translateY(calc(-100% + 250px)) !important; }
                        100% { transform: translateY(0) !important; }
                    }
                `;
                document.head.appendChild(style);
            }
        });
        </script>
        <?php
    }
}